% Adjust radii settings for imfindcircles and preview results

function findCirclesGUI(img)
close all

% Find circles
[centers, radii, metric] = imfindcircles(img,[10 50],'ObjectPolarity', 'bright', 'Sensitivity', 0.95);

decentCircles = find(metric>0.065);
centers = centers(decentCircles, :);
radii = radii(decentCircles);

% Create and position preview
g.f = figure('toolbar','none','menu','none','units','pixels',...
    'position',[0,0,550,500], 'name','Circle finder preview');

% Label circles on image
figure(g.f)
sc(img);
hold on
viscircles(centers, radii,'EdgeColor','b');
movegui(g.f,'center')

% Create and position GUI
h.f = figure('toolbar','none','menu','none','units','pixels',...
    'position',[0,0,550,520], 'name','Spectral Imaging Tool: Circle finder');
movegui(h.f,'west')

% Introduction
h.t(1) = uicontrol('style','text','units','pixels',...
    'position',[15,350,510,150], 'HorizontalAlignment','Left');
intro = sprintf('For the spherical objects segmentation mode, objects are segmented using a circle finding algorithm. This algorithm requires a lower and upper bound radius in pixels.\nClick update preview to overlay the detected circles on the image in the center panel.\nAdjust the upper and lower bounds and update preview until satisfied with the performance of the algorithm.\nClick submit for this image to submit the upper and lower bounds for the current image.\nTo apply these upper and lower bounds to all images being processed, click submit for all images.');
set(h.t(1),'String',intro)

% Options
h.t(2) = uicontrol('style','text','units','pixels',...
    'position',[225,330,500,20],'string','Size range (min, max)', 'HorizontalAlignment','Left');
h.c(1) = uicontrol('style','edit','units','pixels',...
    'position',[225,300,40,15], 'Value',10, 'String','10');
h.c(2) = uicontrol('style','edit','units','pixels',...
    'position',[285,300,40,15],'Value',50, 'String','50');

% Update preview
h.p(1) = uicontrol('style','pushbutton','units','pixels',...
    'position',[200,225,150,20],'string','Update preview',...
    'callback',@p_updatePreview);

% Submit settings and use for all images
h.p(2) = uicontrol('style','pushbutton','units','pixels',...
    'position',[200,150,150,20],'string','Submit for all images',...
    'callback',@p_submitAll);

% Submit settings and use just for this image
h.p(3) = uicontrol('style','pushbutton','units','pixels',...
    'position',[200,75,150,20],'string','Submit for this image',...
    'callback',@p_submitOne);

function p_updatePreview(varargin)
    % Get new size range
    sizeMin = str2double(get(h.c(1),'String'));
    sizeMax = str2double(get(h.c(2),'String'));
    
    % Find circles
    [centers, radii, metric] = imfindcircles(img,[sizeMin sizeMax],'ObjectPolarity', 'bright', 'Sensitivity', 0.95);
    
    decentCircles = find(metric>0.065);
    centers = centers(decentCircles, :);
    radii = radii(decentCircles);
    
    % Relabel circles on image
    figure(g.f)
    clf(g.f)
    sc(img);
    hold on
    viscircles(centers, radii,'EdgeColor','b');
    hold off
    movegui(g.f,'center')
end

function p_submitAll(varargin)
    % Get new size range
    sizeMin = str2double(get(h.c(1),'String'));
    sizeMax = str2double(get(h.c(2),'String'));
    sizeRange = [sizeMin sizeMax];
    assignin('base', 'sizeRange', sizeRange) 
    assignin('base', 'sizeRangeGlobal',1) % global == 1 indicates that settings shold be used for all images being processed
    close all
end

function p_submitOne(varargin)  
    % Get new size range
    sizeMin = str2double(get(h.c(1),'String'));
    sizeMax = str2double(get(h.c(2),'String'));
    sizeRange = [sizeMin sizeMax];
    assignin('base', 'sizeRange', sizeRange) 
    assignin('base', 'sizeRangeGlobal',0) % global == 0 indicates that settings shold be used for just the current image being processed
    close all
end
end